<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package dpii
* @subpackage models
*/

/**
* Parent class for Distribution Lists.
*
* @package dpii
* @subpackage models
*/
class Distribution_list_model extends CI_Model {
	var $_logged_in_user_id;
	public static $form_field_limits = array('list_display_name' => 100, 
											 'list_description' => 250);
	
	public function __construct(){
		parent::__construct();
		$this->load->library('Error_helper', array(), 'error');	
		$this->load->library('Validator', array(), 'is');
		
		$this->_logged_in_user_id = element('user_id', $this->user_model->logged_in_user());	
	}
	
	/////////////////////////////
	// SEARCH
	/////////////////////////////

	/**
	* Retrives distribution lists. based on the given criteria.
	* If no search criteria is given, all lists will be returned.
	*
	* @param array
	* @return array
	*/
	function find($criteria = array()){
		if(!is_array($criteria)) return $this->error->should_be_an_array($criteria);
		return $this->_find($criteria);		
	}
	
	/**
	* Override this in child classes to perform the actual find operation.
	*/
	protected function _find($criteria){
		return $this->error->fatal('Please define a method for '.get_class($this).'::_find()');
	}
	

	/**
	* Looks up a specific distribution list.
	* You may either provide this method with the id of the distribution list or an array of 
	* criteria to search by.  In either case, this method will only return one list.
	*
	* @param scalar|array
	* @return array 
	*/
	function find_one($id_or_criteria){
		if(!is_array($id_or_criteria) && !$this->formatted_like_an_id($id_or_criteria)) return $this->error->should_be_an_id_or_an_array($id_or_criteria);
		return first_element($this->find($this->_criteria_for_find_one($id_or_criteria)));
	}
	
	/** 
	* Hook for child class.
	* The child class will need to identify if the user is attempting to look up by id, and set up the criteria accordingly.
	* @param scalar|array
	* @return array
	*/
	protected function _criteria_for_find_one($id_or_criteria){
		if($this->formatted_like_an_id($id_or_criteria)) 
			return array('id' => $id_or_criteria);
		
		return $id_or_criteria;
	}
	
	/**
	* Checks to see if a list exists with the given id or that matches the search criteria.
	* Child classes may choose to redefine this for a more efficient check (e.g. db count instead of lookup).
	* @param scalar
	* @return boolean
	*/
	function exists($id_or_criteria){
		if(!is_array($id_or_criteria) && !$this->formatted_like_an_id($id_or_criteria)) return $this->error->should_be_an_id_or_search_criteria($id_or_criteria);
		$list = $this->find_one($id_or_criteria);
		return (!empty($list) && is_array($list));
	}
	
	///////////////////////////////////
	// MANAGE LIST C.R.U.D.
	///////////////////////////////////
	
	/**
	* Adds a record for a new distribution list.
	* @param array
	* @return scalar
	*/
	function create($values){
		if(!$this->_values_are_valid_for_create($values)) return false;	
		$id = $this->_create($values);
		if(!$this->formatted_like_an_id($id)){
			
			
			return $this->error->warning('Unable to create a '.humanize(strip_from_end('_model', get_class($this))).' with the following values: '.sprp_for_log($values));
		}
		return $id;
	}
	
	/**
	* Checks that a set of values are valid to be inserted into the database.
	* Child classes should extend this method to include validation specific to that model (e.g. email for global distribution lists).
	*
	* @param array
	* @return boolean
	*/
	protected function _values_are_valid_for_create(&$values){
		if(!is_array($values) || empty($values)) return $this->error->should_be_a_nonempty_array($values);
		
		foreach(array('name', 'description') as $required_value){
			if(!array_key_exists($required_value, $values))
				return $this->error->warning('Please specify a value for '.$required_value);
		}
		
		extract($values);
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!$this->name_is_available($name)) return $this->error->should_be_a_name_that_has_not_been_used($name);
		if(!$this->is->nonempty_string($description)) return $this->error->should_be_a_string($description);
		
		return true;
	}
	
	/**
	* Hook for child classes for the actual create mechanism.
	* Returns true if creation was successful.
	* @param array
	* @return boolean 
	*/
	protected function _create($values){
		return $this->error->fatal('Please define a method for '.get_class($this).'::_create()');
	}
	

	/**
	* Updates a record of an existing distribution list.
	* Returns true if the update was successful.
	* @param array
	* @return boolean
	*/
	function update($id, $values){
		if(!$this->formatted_like_an_id($id)) return $this->error->should_be_an_id($id);
		if(!is_array($values) || empty($values)) return $this->error->should_be_a_nonempty_array($values);
		
		$list = $this->find_one($id);
		if(!is_array($list)) return $this->error->should_be_an_id_for_a_distribution_list($id);
				
		if(!$this->_values_are_valid_for_update($list, $values)) return false;	
		$success = $this->_update($id, $values);
		if(!$success){
			$this->error->warning('Unable to update '.strip_from_end('_model', (get_class($this))).'#'.$id.' with the following values: '.sprp_for_log($values));
		}
		return $success;
	}
	
	/**
	* Checks that a set of values are valid to be inserted into the database.
	* Child classes should extend this method to include validation specific to that model (e.g. email for global distribution lists).
	*
	* @param array
	* @param array
	* @return boolean
	*/	
	protected function _values_are_valid_for_update($list, &$values){
		if(!is_array($values) || empty($values)) return $this->error->should_be_a_nonempty_array($values);
		
		if(array_key_exists('name', $values)){
			if(!$this->is->nonempty_string($values['name'])) return $this->error->should_be_a_nonempty_string($values['name']);
			if($list['name'] != $values['name'] && !$this->name_is_available($values['name'])) return $this->error->should_be_a_name_that_has_not_been_used($name);
		}
		if(array_key_exists('description', $values) && !$this->is->string($values['description']))
			return $this->error->should_be_a_string($values['description']);
		
		return true;
	}
	
	/**
	* Hook for child classes for the actual update mechanism.
	* Returns true if the update was successful.
	* @param array
	* @return boolean 
	*/
	protected function _update($id, $values){
		return $this->error->fatal('Please define a method for '.get_class($this).'::_update()');
	}
	
	/**
	* Deletes a record of a distribution model.
	* Returns true on success.
	* @param scalar
	* @return boolean
	*/
	function delete($id){
		if(!$this->formatted_like_an_id($id)) return $this->error->should_be_an_id($id);
		$list = $this->find_one($id);
		if(!is_array($list)) return $this->error->should_be_an_id_for_a_distribution_list($id);
		$success = $this->_delete($id);
		if(!$success){
			$this->error->warning('Unable to delete '.strip_from_end('_model', get_class($this)).'#'.$id);
		}
		return $success;
	}
	
	/**
	* Hook for child classes for the actual delete mechanism.
	* Returns true if the update was successful.
	* @param scalar
	* @return boolean 
	*/
	protected function _delete($id){
		return $this->error->fatal('Please define a method for '.get_class($this).'::_delete()');
	}
		
	///////////////////////////
	// MANAGE LIST MEMBERSHIP
	///////////////////////////
	
	/**
	* Adds an email address to a distribution list.
	* @param scalar
	* @param string
	* @return boolean
	*/
	function add_address_to_list($list_id, $address){
		if(!$this->formatted_like_an_id($list_id)) return $this->error->should_be_a_distribution_list_id($list_id);
		if(empty($address) || !$this->is->string_like_an_email_address($address)) return $this->error->should_be_an_email_address($address);
		
		$addresses = $this->addresses_for_list($list_id);
		if(!is_array($addresses)) return $this->error->warning('Could not add '.$address.' to '.strip_from_end('_model', get_class($this)).'#'.$list_id);
		if(in_array($address, $addresses)) return true; // doesn't need to be added
		
		$addresses[] = $address;
		sort($addresses); //might as well alphabetize these, it will make life easier
		return $this->update($list_id, array('addresses' => implode(';', $addresses)));
	}
	
	/**
	* Removes an email address from a distribution list.
	* @param scalar
	* @param string
	* @return boolean
	*/
	function remove_address_from_list($list_id, $address){
		if(!$this->formatted_like_an_id($list_id)) return $this->error->should_be_a_distribution_list_id($list_id);
		if(empty($address) || !$this->is->string_like_an_email_address($address)) return $this->error->should_be_an_email_address($address);
		
		$addresses = $this->addresses_for_list($list_id);
		if(!is_array($addresses)) return $this->error->warning('Could not remove '.$address.' from '.strip_from_end('_model', get_class($this)).'#'.$list_id);
		if(!in_array($address, $addresses)) return true; // doesn't need to be removed
		$addresses = array_unique($addresses); //just in case we somehow ended up with duplicates of this address
		unset($addresses[array_search($address, $addresses)]);
		
		return $this->update($list_id, array('addresses' => implode(';', $addresses)));		
	}
	
	/**
	* Returns all of the addresses for a distribution list.
	* @param scalar|array
	* @return array
	*/
	function addresses_for_list($id_or_values){	
		if(!$this->formatted_like_an_id($id_or_values) && !is_array($id_or_values)) return $this->error->should_be_a_distribution_list_id_or_values($id_or_values);
		
		if(is_array($id_or_values))
			$list = $id_or_values;
		else{
			$list = $this->find_one($id_or_values);
			if(empty($list)) return $this->error->should_be_a_distribution_list_id($id_or_values);
		}
		
		if(empty($list['addresses'])) return array();
		return explode(';', $list['addresses']);
	}
	
	function display_names_for_list($id_or_values){
		if(!$this->formatted_like_an_id($id_or_values) && !is_array($id_or_values)) return $this->error->should_be_a_distribution_list_id_or_values($id_or_values);
		
		if(is_array($id_or_values))
			$list = $id_or_values;
		else{
			$list = $this->find_one($id_or_values);
			if(empty($list)) return $this->error->should_be_a_distribution_list_id($id_or_values);
		}	
	
		$addresses = $this->addresses_for_list($list);
		
		//defaulting to using the address as the display name on external addresses	
		$display_names = array_combine($addresses, $addresses);
		$display_names = array_merge($display_names, $this->_display_names_for_external_addresses($list['id'], $addresses));
		$display_names = array_merge($display_names, $this->display_names_for_direct_addresses($addresses));
		return $display_names;
	}
	
	function display_names_for_direct_addresses($addresses){
		if(!is_array($addresses)) return $this->error->should_be_an_array($addresses);
		
		$direct_addresses = $this->direct_addresses_for_list($addresses);
		if(empty($direct_addresses)) return array();
		
		$filter = '(mail='.implode(')(mail=', $direct_addresses).')';
		if(count($direct_addresses) > 1)
			$filter = '(|'.$filter.')';
		$filter = '(&(objectclass=inetOrgPerson)'.$filter.')';

		$results = ldap_search( $this->ldap->conn, LDAP_ACCOUNTS_DN, $filter);
		$entries = $this->ldap->get_formatted_entries($results, 'mail');	
		
		return collect('displayname', $entries);	
	}
	
	//by default, this will return the addresses; extend in child classes to use the appropriate method.
	protected function _display_names_for_external_addresses($list_id, $addresses){
		if(!is_array($addresses)) return $this->error->should_be_an_array($addresses);
		$external_addresses = $this->external_addresses_for_list($addresses);
		return array_combine($external_addresses, $external_addresses);
	}
		
	
	/**
	* Returns just the external addresses for a list.
	* You may pass this function the list id, the addresses as an array, or the addresses as a string (separted by semi-colons).
	*
	* @param scalar|string|array
	* @param string
	* @return array
	*/
	function external_addresses_for_list($list_id_or_addresses){
		if(!$this->formatted_like_an_id($list_id_or_addresses) && !is_string($list_id_or_addresses) && !is_array($list_id_or_addresses))
			return $this->error->should_be_a_list_id_or_list_of_addresses($list_id_or_addresses);
		
		//check to see if we were given a list id
		if($this->formatted_like_an_id($list_id_or_addresses)){
			$addresses = $this->addresses_for_list($list_id_or_addresses);
			if(!is_array($addresses)) return $this->error->should_be_a_list_id($list_id_or_addresses);
		}
		
		//for strings or arrays, there's no point in parsing further if they're empty
		if(empty($list_id_or_addresses)) return array();
		if(is_string($list_id_or_addresses)) $addresses = explode(';', $list_id_or_addresses);
		if(is_array($list_id_or_addresses)) $addresses = $list_id_or_addresses;
		
		return array_filter($addresses, array($this, 'formatted_like_an_external_address'));
	}
	
	/**
	* Returns just the Direct addresses for a list.
	* You may pass this function the list id, the addresses as an array, or the addresses as a string (separted by semi-colons).
	*
	* @param scalar|string|array
	* @param string
	* @return array
	*/
	function direct_addresses_for_list($list_id_or_addresses){
		if(!$this->formatted_like_an_id($list_id_or_addresses) && !is_string($list_id_or_addresses) && !is_array($list_id_or_addresses))
			return $this->error->should_be_a_list_id_or_list_of_addresses($list_id_or_addresses);
		
		//check to see if we were given a list id
		if($this->formatted_like_an_id($list_id_or_addresses)){
			$addresses = $this->addresses_for_list($list_id_or_addresses);
			if(!is_array($addresses)) return $this->error->should_be_a_list_id($list_id_or_addresses);
		}
		
		//for strings or arrays, there's no point in parsing further if they're empty
		if(empty($list_id_or_addresses)) return array();
		if(is_string($list_id_or_addresses)) $addresses = explode(';', $list_id_or_addresses);
		if(is_array($list_id_or_addresses)) $addresses = $list_id_or_addresses;
		
		return array_filter($addresses, array($this, 'formatted_like_a_direct_address'));
	}
	
	/**
	* Retrieve information about each user for this list from the db & ldap.
	* Returns an array of user data, keyed by email address: e.g. array('vuser1@dev.careinbox.com => array('display_name' => 'Valid User')
	*
	* @todo - do we need the db information on this?  might want to simplify & just look up the display names.
	*
	* @param scalar|strings|array
	* @return array
	*/	
	function users_for_list($list_id_or_addresses){
		$direct_addresses = $this->direct_addresses_for_list($list_id_or_addresses);
		if(!is_array($direct_addresses)) return false;
		if(empty($direct_addresses)) return array();
		sort($direct_addresses);
		$direct_addresses = array_map('strtolower', $direct_addresses); //normalize addresses
		
		$usernames = array_map( 'strip_from_end', array_fill( 0, count($direct_addresses), '@'.DIRECT_DOMAIN), $direct_addresses);
		$results = $this->db->where_in('user_name', $usernames)->order_by('user_name')->get('users')->result_array();
		$users = array_combine($direct_addresses, $results);
		
		$this->load->library('ldap');
		$filter = "(&(ObjectClass=person)(|(mail=" . implode(")(mail=", $direct_addresses) . ")))";
		$ldap_entries = $this->ldap->search($search = null, $sizelimit = NULL, $properties = NULL, $filter);
		foreach($ldap_entries as $entry){
			if(array_key_exists('mail', $entry)){
				$users[strtolower($entry['mail'])]['display_name'] = $entry['displayname'];
			}
		}
		
		return $users;
	}
	
	/**
	* A unique identifier for this list that may be used in recipient strings.
	* This needs to be easily distinguished from email addresses and from other list types.
	* It also needs to be easily converted back to the list id.
	* @param scalar
	* @return string
	*/
	function alias($list_id){
		return $this->error->fatal('Please define a method for '.get_class($this).'::alias()');
	}
	
	/**
	* Obtains the list id from the list alias.
	* @see alias()
	* @param string
	* @return scalar
	*/
	function id_from_alias($alias){
		return $this->error->fatal('Please define a method for '.get_class($this).'::id_from_alias()');
	}
	
	
	/**
	* Parses a recipient string (to/cc/bcc) for any distribution lists and substitutes in the member addresses.
	* Note that this method does not currently preserve the original recipient order for the message.
	* @todo OK that we're not preserving the recipient order?   Could alphabetize, but that might confuse people further.
	* @param string
	* @return string
	*/
	function substitute_addresses_for_aliases($recipients){
		if(!is_string($recipients)) return $this->error->should_be_a_string($recipients);
		if(empty($recipients)) return '';
		
		$addresses = array_map('trim', array_unique(explode(';', $recipients)));
		$lists = array_filter($addresses, array($this, 'formatted_like_an_alias'));
		foreach($lists as $alias){
			//first, check to make sure it really is an alias. skip if not.
			$id = $this->id_from_alias($alias);
			if(!$this->exists($id)) continue; 
			
			//append the addresses to the end. 
			unset($addresses[array_search($alias, $addresses)]);
			$addresses = array_merge($addresses, $this->addresses_for_list($id));
		}
		$addresses = array_unique($addresses);
		return implode(';', $addresses);
	}
	
	
	////////////////////////
	// VALIDATION 
	////////////////////////
	
	/**
	* True if the scalar matches the unique identifier for this list type.
	* For private lists, this will be the db id, and in LDAP this will be the uid.
	*
	* @param scalar
	* @return boolean
	*/
	function formatted_like_an_id($id){
		return $this->is->nonzero_unsigned_integer($id);
	}
	
	/**
	* True if the string matches the format of {@link alias()}.
	* @param string
	* @return boolean
	*/
	function formatted_like_an_alias($alias){
		return $this->error->fatal('Please define a method for '.get_class($this).'::formatted_like_an_alias()');
	}	
	
	/**
	* Checks to see if an address is formatted like a Direct address.
	* N.B. - This is intended to be used to distingush between Direct & external addresses in lists, and to save overhead,
	* it doesn't check the database to make sure that there really is a user with this email address.  You should check 
	* the database if you need to verify that a user actually exists.
	* @param string
	* @return boolean
	*/
	function formatted_like_a_direct_address($address){
		if(!$this->is->nonempty_string($address)) return $this->error->should_be_a_nonempty_string($address);
		if(!$this->is->string_like_an_email_address($address)) return false;	
		return string_ends_with('@'.DIRECT_DOMAIN, $address);
	}
	
	/**
	* True if the string is formatted like an email address but does NOT belong to the Direct domain.
	* @param string
	* @return boolean
	*/
	function formatted_like_an_external_address($address){
		if(!$this->is->nonempty_string($address)) return $this->error->should_be_a_nonempty_string($address);
		if(!$this->is->string_like_an_email_address($address)) return false;
		return !$this->formatted_like_a_direct_address($address);
	}
	
	/**
	* Check to make sure that this name isn't in use.
	* For private lists, this will be automatically limited to the user's list, since all find/exist
	* searchs are limited to the user's lists.
	* @param string
	* @return boolean
	*/
	function name_is_available($name){
		if(!$this->is->nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		$conditions = array('name' => $name);
		return !$this->exists($conditions);
	}
	
	/**
	* ID for the currently logged in user.
	* @return int
	*/
	function logged_in_user_id(){
		return $this->_logged_in_user_id;	
	}	
	
}
